//#include "system.h"
#include  "device.h"
#include "F021_F28003x_C28x.h"
#include "r_eeprom.h"
#define  EE_FREQ_MHZ    ((uint32_t)120)

///  Header/marker of a free EEPROM page (erased state)
#define  EE_FREE        ((uint16_t)0xFFFFu)

#define  EEPROM_ALIGNMENT   __attribute__((aligned (4)))

#define  EEPROM_DATA_ALIGNMENT   __attribute__((aligned (8)))


///  Virtual EEPROM page header
typedef struct {
    uint16_t    marker;     ///< EEPROM block marker
    uint16_t    reserved;   ///< not used, reserved
    uint32_t    check;      ///< block check value

} EEPROM_ALIGNMENT EEPROMHeader_t;

#define EE_ARRAY rs_eeprom
#define EE_BUFFER rs_eeprom_buffer
#define EE_USED RS_EE_VALID
typedef struct {
    EEPROMHeader_t header;
    RsConfig_t data;
} EEPROM_ALIGNMENT EEPROM_t;
#define EE_BANK (Fapi_FlashBank2)
#define EE_SECTOR (14u)
#ifdef EEPROM_SIMULATE_WRITE
#define EE_SECTION "ramgs1"
#else
#define EE_SECTION "eeprom1"
#endif
#ifdef EEPROM_SIMULATE_WRITE
#define SIZE_SECTOR (0x100u)
#else
#define SIZE_SECTOR (0x1000u)
#endif
#define EE_NUMPAGES (SIZE_SECTOR / sizeof(EEPROM_t))
#pragma DATA_SECTION(EE_ARRAY, EE_SECTION)
static EEPROM_t EE_ARRAY[EE_NUMPAGES];
static EEPROM_t EEPROM_DATA_ALIGNMENT EE_BUFFER;
static uint32_t update_check(uint32_t check_in, uint32_t data);
static uint32_t rs_validate(void);
static bool check_status(Fapi_StatusType fapi_status);
static uint16_t find_last(bool as_used);
static bool sector_erase(void);
static bool write_page(uint16_t index);


void main_tasks(void)
{

}

bool rs_eeprom_init( void )
{
    bool is_valid;
    Fapi_StatusType fapi_status;
    
    fapi_status = Fapi_initializeAPI(F021_CPU0_BASE_ADDRESS, EE_FREQ_MHZ);
    if (fapi_status == Fapi_Status_Success)
    {
        is_valid = true;
    }
    else
    {
        is_valid = false;
    }
    if (is_valid)
    {
        Flash_disablePrefetch((uint32_t)FLASH0CTRL_BASE);
        FLASH_DELAY_CONFIG;
        fapi_status = Fapi_setActiveFlashBank(EE_BANK);
        if (fapi_status != Fapi_Status_Success)
        {
            is_valid = false;
        }
        Flash_enablePrefetch((uint32_t)FLASH0CTRL_BASE);
        FLASH_DELAY_CONFIG;
    }
    return is_valid;
}

bool rs_eeprom_read(RsConfig_t * pconfig)
{
    bool is_valid = false;
    uint16_t index;
    index = find_last(true);
    if (index < EE_NUMPAGES)
    {
        EE_BUFFER = EE_ARRAY[index];
        uint32_t check = rs_validate();
        if (check == EE_ARRAY[index].header.check)
        {
            *pconfig = EE_BUFFER.data;
            is_valid = true;
        }
    }
    return is_valid;
}

bool rs_eeprom_write(RsConfig_t * pconfig)
{
    bool is_valid;
    uint16_t index;
    EE_BUFFER.data = *pconfig;
    index = find_last(false);
    if (index >= EE_NUMPAGES)
    {
        is_valid = sector_erase();
        index = 0;
    }
    else
    {
        is_valid = true;
    }
    if (is_valid)
    {
        is_valid = (EE_ARRAY[index].header.marker == EE_FREE)
                    ? write_page(index)
                    : false;
    }
    return is_valid;
}

static uint32_t update_check(uint32_t check_in, uint32_t data)
{
    uint32_t check_out =
        data + (check_in << 6u) + (check_in << 16u) - check_in;
    return check_out;
}

static uint32_t rs_validate (void)
{
    uint32_t calc_check = 0xFFFFFFFFu;
    uint32_t u32_data;
    calc_check = update_check(calc_check, EE_BUFFER.header.marker);
    calc_check = update_check(calc_check, (uint32_t)EE_BUFFER.data.e_type);
    calc_check = update_check(calc_check, (uint32_t)EE_BUFFER.data.s_control);
    calc_check = update_check(calc_check, (uint32_t)EE_BUFFER.data.o_control);
    u32_data = (uint32_t)((uint64_t)EE_BUFFER.data.o_limit >> 32);
    calc_check = update_check(calc_check, u32_data);
    u32_data = (uint32_t)((uint64_t)EE_BUFFER.data.o_limit & 0xFFFFFFFFu);
    calc_check = update_check(calc_check, u32_data);
    calc_check = update_check(calc_check, EE_BUFFER.data.pm_time);
    return calc_check;
}

static uint16_t find_last(bool as_used)
{
    uint16_t index;
    uint16_t left = 0u;
    uint16_t right = EE_NUMPAGES - 1u;
    while (left < right)
    {
        uint16_t pos = (left + right) / 2u;
        if (EE_ARRAY[pos].header.marker == EE_FREE)
        {
            right = pos - 1u;
        }
        else
        {
            left = pos + 1u;
        }
    }
    index = left;
    if (as_used)
    {
        if (EE_ARRAY[index].header.marker != EE_USED)
        {
            index = EE_NUMPAGES;
        }
    }
    else
    {
        if (EE_ARRAY[index].header.marker != EE_FREE)
        {
            index += 1u;
        }
    }
    return index;
}

static inline bool sector_erase(void)
{
    bool is_valid = true;
    Fapi_StatusType fapi_status;
    uint16_t wr_protect = (uint16_t)(0xFFFFu ^ (1u << EE_SECTOR));
#ifdef EEPROM_SIMULATE_WRITE
    uint16_t index = 0u;
    do {
        EE_ARRAY[index].header.marker = (uint16_t)0xFFFFu;
        EE_ARRAY[index].header.reserved = wr_protect;
        index++ ;
    } while (index < EE_NUMPAGES);
    fapi_status = Fapi_checkFsmForReady();
#else
    void * v_address = EE_ARRAY;
    uint32_t * u_address = (uint32_t *)v_address;
    fapi_status = Fapi_issueBankEraseCommand(u_address, wr_protect);
#endif
    is_valid = check_status(fapi_status);
    if (is_valid)
    {
#ifdef EEPROM_SIMULATE_WRITE
#else
        Fapi_StatusType fapi_status;
        Fapi_FlashStatusWordType status_word;
        fapi_status = Fapi_doBlankCheck(u_address,
                                        SIZE_SECTOR / 2u,
                                        &status_word);
        (void) status_word;
        if (fapi_status != Fapi_Status_Success)
        {
            is_valid = false;
        }
#endif
    }
    return is_valid;
}

static bool check_status(Fapi_StatusType fapi_status)
{
    bool result;
    uint32_t retries = 2000000u;
    while ((Fapi_checkFsmForReady() != Fapi_Status_FsmReady)
           && (retries > 0u))
    {
        main_tasks();
        retries--;
    }
    Fapi_FlashStatusType fsm_status = Fapi_getFsmStatus();
    if ((fapi_status != Fapi_Status_Success)
        || (fsm_status != 0u)
        || (retries == 0u))
    {
        result = false;
    }
    else
    {
        result = true;
    }
    return result;
}

static bool write_page(uint16_t index)
{
    bool is_valid;
    Fapi_StatusType fapi_status;
    EE_BUFFER.header.marker = EE_USED;
    EE_BUFFER.header.check = rs_validate();
#ifdef EEPROM_SIMULATE_WRITE
    EE_BUFFER.header.reserved = sizeof(EE_BUFFER);
    EE_ARRAY[index] = EE_BUFFER;
    fapi_status = Fapi_checkFsmForReady();
#else
    void * v_address = &EE_ARRAY[index];
    void * v_page = &EE_BUFFER;
    struct {
        uint32_t * address;
        uint16_t * data_page;
        uint16_t data_size;
    } eeinfo;
    eeinfo.address = (uint32_t *)v_address;
    eeinfo.data_page = (uint16_t *)v_page;
    eeinfo.data_size = sizeof(EE_BUFFER);
    uint16_t remains = eeinfo.data_size;
    do {
        fapi_status =
            Fapi_issueProgrammingCommand(eeinfo.address,
                                         eeinfo.data_page,
                                         4,
                                         NULL, 0u,
                                         Fapi_AutoEccGeneration);
        is_valid = check_status(fapi_status);
        if (is_valid)
        {
            Fapi_FlashStatusWordType status_word;
            fapi_status =
                Fapi_doVerify(eeinfo.address,
                              2u,
                              (uint32_t *)eeinfo.data_page,
                              &status_word);
            (void) status_word;
            if (fapi_status != Fapi_Status_Success)
            {
                is_valid = false;
            }
        }
        eeinfo.address += 2u;
        eeinfo.data_page += 4u;
        remains -= 4u;
    } while (is_valid && (remains != 0u));
#endif
    return is_valid;
}
